<?php

namespace App\Http\Controllers;

use App\Models\Gym;
use Illuminate\Http\Request;

class GymController extends Controller
{
    public function liste(Request $request)
    {
        // Récupérer tous les spécialités uniques pour le filtre
        $specialites = Gym::where('validate', true)
            ->distinct('specialite')
            ->pluck('specialite')
            ->filter()
            ->values();

        // Récupérer toutes les nationalités uniques pour le filtre
        $nationalites = Gym::where('validate', true)
            ->distinct('nationalite')
            ->pluck('nationalite')
            ->filter()
            ->values();

        // Commencer avec la requête de base
        $query = Gym::where('validate', true);

        // Filtre par spécialité
        if ($request->has('specialite') && !empty($request->specialite)) {
            $query->where('specialite', $request->specialite);
        }

        // Filtre par nationalité
        if ($request->has('nationalite') && !empty($request->nationalite)) {
            $query->where('nationalite', $request->nationalite);
        }

        // Recherche par informations/adresse/description
        if ($request->has('search') && !empty($request->search)) {
            $searchTerm = $request->search;
            $query->where(function($q) use ($searchTerm) {
                $q->where('informations', 'LIKE', "%{$searchTerm}%")
                    ->orWhere('description', 'LIKE', "%{$searchTerm}%")
                    ->orWhere('adresse', 'LIKE', "%{$searchTerm}%")
                    ->orWhere('tel', 'LIKE', "%{$searchTerm}%")
                    ->orWhere('email', 'LIKE', "%{$searchTerm}%");
            });
        }

        // Vérifier si un filtre a été appliqué
        $hasFilter = $request->has('search') && !empty($request->search) ||
            $request->has('specialite') && !empty($request->specialite) ||
            $request->has('nationalite') && !empty($request->nationalite);

        // Récupérer les gyms avec pagination
        $gyms = $query->paginate(12)->appends($request->except('page'));

        return view('gyms.liste', [
            'gyms' => $gyms,
            'specialites' => $specialites,
            'nationalites' => $nationalites,
            'hasFilter' => $hasFilter
        ]);
    }

    public function create()
    {
        return view('gyms.create');
    }

    public function store(Request $request)
    {
        // Validation des données
        $request->validate([
            'photo1' => 'nullable|image|mimes:jpg,jpeg,png,gif,webp|max:2048',
            'photo2' => 'nullable|image|mimes:jpg,jpeg,png,gif,webp|max:2048',
            'photo3' => 'nullable|image|mimes:jpg,jpeg,png,gif,webp|max:2048',
            'informations' => 'required|string|max:255',
            'adresse' => 'required|string|max:255',
            'tel' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'website' => 'nullable|string|max:255|url',
            'specialite' => 'nullable|string|max:255',
            'nationalite' => 'required|string|max:255',
            'description' => 'required|string|max:400',
        ], [
            'photo1.max' => 'La photo 1 ne doit pas dépasser 2MB',
            'photo2.max' => 'La photo 2 ne doit pas dépasser 2MB',
            'photo3.max' => 'La photo 3 ne doit pas dépasser 2MB',
            'informations.required' => 'Le nom du gym est requis',
            'adresse.required' => 'L\'adresse est requise',
            'tel.required' => 'Le téléphone est requis',
            'nationalite.required' => 'La nationalité est requise',
            'description.required' => 'La description est requise',
            'description.max' => 'La description ne doit pas dépasser 400 caractères',
            'website.url' => 'Le site web doit être une URL valide (commence par http:// ou https://)',
        ]);

        // Préparer les données
        $data = $request->except(['photo1', 'photo2', 'photo3']);

        // Pour les visiteurs, le gym est en attente de validation par défaut
        $data['validate'] = false;

        // Traitement des images en BLOB
        foreach (['photo1', 'photo2', 'photo3'] as $photo) {
            if ($request->hasFile($photo)) {
                // Lire le contenu du fichier en binaire
                $imageData = file_get_contents($request->file($photo)->getRealPath());
                $data[$photo] = $imageData;
            }
        }

        // Création du gym
        Gym::create($data);

        // Message de succès avec redirection
        return redirect()->route('gyms.create')->with('success', 'Votre gym a été enregistré avec succès ! Il sera visible après validation par l\'administration.');
    }
}